/*
 * @brief Secondary loader host interface support
 *
 * @note
 * Copyright(C) NXP Semiconductors, 2014
 * All rights reserved.
 *
 * @par
 * Software that is described herein is for illustrative purposes only
 * which provides customers with programming information regarding the
 * LPC products.  This software is supplied "AS IS" without any warranties of
 * any kind, and NXP Semiconductors and its licensor disclaim any and
 * all warranties, express or implied, including all implied warranties of
 * merchantability, fitness for a particular purpose and non-infringement of
 * intellectual property rights.  NXP Semiconductors assumes no responsibility
 * or liability for the use of the software, conveys no license or rights under any
 * patent, copyright, mask work right, or any other intellectual property rights in
 * or to any products. NXP Semiconductors reserves the right to make changes
 * in the software without notification. NXP Semiconductors also makes no
 * representation or warranty that such application will be suitable for the
 * specified use without further testing or modification.
 *
 * @par
 * Permission to use, copy, modify, and distribute this software and its
 * documentation is hereby granted, under NXP Semiconductors' and its
 * licensor's relevant copyrights in the software, without fee, provided that it
 * is used in conjunction with NXP Semiconductors microcontrollers.  This
 * copyright, permission, and disclaimer notice must appear in all copies of
 * this code.
 */

#ifndef SL_HOSTIF_H
#define SL_HOSTIF_H

#include "chip.h"
#include "LPC865.h"

#ifdef __cplusplus
extern "C" {
#endif

#define SL_HOSTIFBUFFSIZE                   (SL_FLASH_BLOCK_SZ + 32)/* buffer to fit FLASH block and parameteres */

/* Protocol commands */
#define CFG_BOOT_WHO_AM_I                   0x55

/************************************************
    Secondary boot loader version history.
   02_18_2015: Fixing implemented to bypass hostif detection when there is valid image header for SPI hostinterface.
    Autodetection is still happening with I2C host interface even if image is already programmed.
    This is needed as I2C pin assignments do not exist in the image header. This works because for I2C hostIF autodetection,
    the first packet is not lost.
   0.3    Fixed issue with random command in SPI host interface mode.
   0.2    Added sub-block commands.
        Added nIRQ pin check for IMG_NO_CRC image type.
   0.1    Initial version.
 **************************************************/
#define CFG_MAJOR_VERSION                   0x00
#define CFG_MINOR_VERSION                   0x03

#define CRP_ADDR            0x2FC
#define Get_CRP()           (*(uint32_t *) CRP_ADDR)
#define CRP_PAGE_ADDR       (CRP_ADDR & ~(SL_FLASH_PAGE_SZ - 1))
#define CRP_PAGE_NUM        (CRP_PAGE_ADDR / SL_FLASH_PAGE_SZ)
#define CRP_SECT_NUM        (CRP_PAGE_ADDR / SL_FLASH_SECT_SZ)
/* CRP VALUE-2 that will be written when enable secure command is issued */
#define CRP2_VALUE   0x87654321

/* Structure defining a packet of data received from the host */
typedef struct {
    uint32_t    bytes;
    uint8_t     buffer[SL_HOSTIFBUFFSIZE];
} SL_HOSTIF_PACKET_T;
extern SL_HOSTIF_PACKET_T sendBuff, recvBuff;

/* Initialize host interface subsystem */
void Hostif_Init(void);

/* Port and pin seelction and restoration structure */
typedef struct {
    uint8_t     port;                    /* Port mapped to the signal */
    uint8_t     pin;                     /* Pin mapped to the signal */
} SL_PORTPIN_T;

/*Configurate IRQ pin, P0_15*/
void IRQ_Pin_Config(SL_PORTPIN_T *pPortInfo);

/* Parses port and pin data from a byte */
void parsePortData(SL_PORTPIN_T *pPortInfo, uint8_t packetByte);

/* A packet received from the host is submitted to this function. When the packet
   is submitted, IRQs will be disabled. The return information to the host is
   setup via the tohost structure (which is preallocated prior to the call). */
void processHostIfPacket(SL_HOSTIF_PACKET_T *pFromHost, SL_HOSTIF_PACKET_T *pToHost);

/* Parses the initial packet and determines pin mapping based on the data */
int parsePacketA5(SL_HOSTIF_PACKET_T *pFromHost, int packetFromSPI);

/* Assert interrupt line to Host/AP */
void Hostif_AssertIRQ(void);

/* De-assert interrupt line to Host/AP */
void Hostif_DeAssertIRQ(void);

/* Is a host packet send or reeive done? */
bool hostIfPacketPending(void);

/* Sets up pinmuxing for an I2C interface */
void setupMuxingI2C(int i2cNum);

/* Sets up a I2C slave interface */
void setupInterfaceI2C(int i2cNum);

/* Starts I2C host interface processing */
void startHostIfI2C(int i2cNum);

/* I2C host interface processing loop */
void loopHostIfI2C(int i2cNum);

/* Shut down a I2C slave interface */
void shutdownInterfaceI2C(int i2cNum);

/* Sets up pinmuxing for an SPI interface */
void setupMuxingSPI(int spiNum);

/* Sets up pinmuxing for an SPI MISO pin only */
void setupMuxingSPIMiso(int spiNum);

/* Sets up a SPI slave interface */
void setupInterfaceSPI(int spiNum);

/* Starts SPI host interface processing */
void startHostIfSPI(void);

/* SPI host interface processing loop */
void loopHostIfSPI(int spiNum);

/* Shut down a SPI slave interface */
void shutdownInterfaceSPI(int spiNum);

/* Structure that hols the selected host interface configuration */
typedef struct {
    int                 selectedHostSpi;    /* 0 = I2C, 1 = SPI selected */
    volatile uint32_t   ifSel;                /* Interface selection for host (SL_IFSEL_T) */
    I2C_Type           *pI2C;                /* Valid if selectedHostSpi == 1 */
    SPI_Type           *pSPI;                /* Valid if selectedHostSpi == 0 */
    SL_PORTPIN_T        hostIrqPortPin;        /* Host obtained IRQ port and pin mapping */
    SL_PORTPIN_T        hostSckPortPin;        /* Mapped SPI clock pin mapping */
    SL_PORTPIN_T        hostMisoPortPin;    /* Mapped SPI MOSI pin mapping */
    SL_PORTPIN_T        hostMosiPortPin;    /* Mapped SPI MISO pin mapping */
    SL_PORTPIN_T        hostSselPortPin;    /* Mapped SPI SSEL pin mapping */
} SL_IFCONFIG_T;
extern SL_IFCONFIG_T slIfConfig;

#ifdef __cplusplus
}
#endif

#endif /* SL_HOSTIF_H */
